<?php
namespace WPUmbrella\Core\Restore\ChainResponsibility;

if (!defined('ABSPATH')) {
    exit;
}

use WPUmbrella\Core\Restore\ChainResponsibility\RestoreProcessHandler;
use WPUmbrella\Core\Restore\Memento\CaretakerHandler;
use WPUmbrella\Core\Restore\Memento\RestoreOriginator;

class DownloadBuildZipHandler extends RestoreProcessHandler implements CaretakerHandler
{

   /**
	 * @param $data array
	 *
	 * $data = [
	 * 		'url' => '',
	 * 		'logfile' => '',
	 * ]
	 *
	 * @param $type string ('files'|'database')
	 *
	 */
	protected function download($data, $type){

		wp_umbrella_get_service('LogProcess')->log(sprintf('Download zip %s', $type), [
			'logfile' => $data['logfile'],
		]);

		$timeStart = microtime(true);

		$file = $type === 'files' ? 'backup.zip' : 'database.zip';
		$filename = sprintf('%s/%s', untrailingslashit(WP_UMBRELLA_DIR_TEMP_RESTORE), $file);
		$result = wp_umbrella_get_service('DownloadZip')->handle($data['url'], $filename);

		wp_umbrella_get_service('LogProcess')->log(sprintf('Time download %s', (microtime(true) - $timeStart)), [
			'logfile' => $data['logfile'],
		]);

		if (!$result) {
			return [
				'success' => false,
				'error_code' => sprintf('download_build_zip_%s', $type),
			];
		}

		return [
			'success' => true,
		];
	}

	public function getTypeAndUrlDownload($originator){
		$url = $originator->getValueInState('url_download_backup_files');
		$zipFilesPath = $originator->getValueInState('zip_files_path');

		if(!empty($url) && empty($zipFilesPath)){
			return [
				'url' => $url,
				'type' => 'files',
				'path_state' => 'zip_files_path',
				'zip_name' => 'backup.zip'
			];
		}


		$url = $originator->getValueInState('url_download_backup_database');
		$zipDatabasePath = $originator->getValueInState('zip_database_path');

		if(!empty($url) && empty($zipDatabasePath)){
			return [
				'url' => $url,
				'type' => 'database',
				'path_state' => 'zip_database_path',
				'zip_name' => 'database.zip'
			];
		}

		return null;

	}

	/**
	 * This function handles the case of the 2 downloads but will execute only one restoration
	 */
	public function handle($data)
    {
        $originator = $this->getOriginatorByData($data);
		$infos = $this->getTypeAndUrlDownload($originator);

		$type = $infos['type'];
		$url = $infos['url'];

		if($type === null){
			$this->setFailHandler($data, [
				'error_code' => 'download_build_zip_files_no_type',
			]);
			return false;
		}

		$logfile = $this->getLogfile($originator->getState());

		try {
			if(!empty($url)){
				$response = $this->download([
					'logfile' => $logfile,
					'url' => $url,
				], $type);

				if(!$response['success']){
					$this->setFailHandler($data, [
						'error_code' => $result['error_code'],
					]);
					return false;
				}

				$originator->setValueInState($infos['path_state'], sprintf('%s/%s', untrailingslashit(WP_UMBRELLA_DIR_TEMP_RESTORE), $infos['zip_name']));
			}

		} catch (\Exception $e) {
			$this->setFailHandler($data, [
				'error_code' => 'download_build_zip_files',
				'error_message' => $e->getMessage()
			]);
			return false;
		}


		wp_umbrella_get_service('LogProcess')->log('Finish download', [
			'logfile' => $logfile,
		]);

		$restorationId = $originator->getValueInState('restoration_id');

		wp_umbrella_get_service('RestorationApi')->postFinishDownloadZip($restorationId, [
			"type" => $type,
		]);

		/**
		 * Call queue for restore
		 */
		if($type === 'files'){
			wp_umbrella_get_service('RestoreRunner')->scheduledRestorePrepareFiles();
		}
		else if($type === 'database'){
			wp_umbrella_get_service('RestoreRunner')->scheduledRestorePrepareDatabase();
		}


        $data['originator'] = $originator;

        return parent::handle($data);
    }
}
