<?php
namespace WPUmbrella\Actions\Admin\Ajax;


use WPUmbrella\Core\Hooks\ExecuteHooksBackend;

class ValidationApiKey implements ExecuteHooksBackend
{
	protected $optionService;

	protected $getOwnerService;

    public function __construct()
    {
        $this->optionService = \wp_umbrella_get_service('Option');
        $this->getOwnerService = wp_umbrella_get_service('Owner');
    }

    public function hooks()
    {
        add_action('wp_ajax_wp_umbrella_valid_api_key', [$this, 'validate']);
        add_action('wp_ajax_wp_umbrella_check_api_key', [$this, 'check']);
    }

    public function validate()
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error([
                'code' => 'not_authorized',
            ]);
            exit;
        }

        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'wp_umbrella_valid_api_key')) {
            wp_send_json_error([
                'code' => 'not_authorized',
            ]);
            exit;
        }

        if (!isset($_POST['api_key'])) {
            wp_send_json_error([
                'code' => 'missing_parameters',
            ]);
            exit;
        }

        $apiKey = sanitize_text_field($_POST['api_key']);

        $options['allowed'] = false;
        if (empty($apiKey)) {
            wp_send_json_error([
                'code' => 'missing_parameters',
            ]);
            exit;
        }

        $optionsBdd = $this->optionService->getOptions();
        $newOptions = wp_parse_args($options, $optionsBdd);

        try {
            $data = $this->getOwnerService->validateApiKeyOnApplication($apiKey);

            if (isset($data['code'])) {
                $newOptions['allowed'] = false;
                $newOptions['api_key'] = '';
                $this->optionService->setOptions($newOptions);

                wp_send_json_error([
                    'code' => 'api_key_invalid',
                ]);
                return;
            }

			if(!isset($data['result'])){
				wp_send_json_success([
                    'user' => null,
                    'api_key' => $apiKey,
                    'project_id' => null
                ]);

                return;
			}

            $owner = $data['result'];

			if(isset($owner['total_projects']) && isset($owner['limit_projects'])){
				if($owner['total_projects'] >= $owner['limit_projects']){
					wp_send_json_success([
						'user' => $owner,
						'api_key' => $apiKey,
						'projects' => null,
						'code' => 'limit_excedeed',
					]);
					return;
				}
			}

			// Project exist, need to regenerate secret token
            if ($data && !isset($data['code']) && isset($owner['project']['id'])) {
                $newOptions['allowed'] = true;
                $newOptions['api_key'] = $apiKey;
                $newOptions['project_id'] = $owner['project']['id'];

				$secretToken = wp_umbrella_generate_random_string(128);
				$newOptions['secret_token'] = $secretToken;
				$this->optionService->setOptions($newOptions);

				$responseValidateSecret = wp_umbrella_get_service('Projects')->validateSecretToken([
					'base_url' => site_url(),
					'rest_url' => rest_url(),
					'secret_token' => $secretToken
				], $apiKey);

				if(!is_array($responseValidateSecret) || !isset($responseValidateSecret['success'])){
					$newOptions['allowed'] = false;
					$newOptions['api_key'] = '';
					$newOptions['project_id'] = '';
					$newOptions['secret_token'] = '';
					$this->optionService->setOptions($newOptions);

					wp_send_json_error([
						'code' => 'not_authorized',
					]);
					return;
				}

				if(!$responseValidateSecret['success']){
					$newOptions['allowed'] = false;
					$newOptions['api_key'] = '';
					$newOptions['project_id'] = '';
					$newOptions['secret_token'] = '';
					$this->optionService->setOptions($newOptions);

					wp_send_json_error([
						'code' => 'not_authorized',
					]);
					return;
				}

                $this->optionService->setOptions($newOptions);

                wp_send_json_success([
                    'user' => $owner,
                    'api_key' => $apiKey,
                    'project_id' => $newOptions['project_id']
                ]);

                return;
			// Api is valid but project not exist
            } elseif (!isset($owner['project']['id'])) {
                $newOptions['allowed'] = false;
                $newOptions['api_key'] = $apiKey;

                $this->optionService->setOptions($newOptions);

                wp_send_json_success([
                    'user' => $owner,
                    'api_key' => $apiKey,
                    'code' => 'project_not_exist',
                ]);
            }
        } catch (\Exception $e) {
            \wp_umbrella_get_service('Logger')->error($e->getMessage());
            wp_send_json_error([
                'code' => 'unknown_error',
            ]);
            exit;
        }
    }

    public function check()
    {
        if (!current_user_can('manage_options')) {
            wp_send_json_error([
                'code' => 'not_authorized',
            ]);
            exit;
        }

        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'wp_umbrella_check_api_key')) {
            wp_send_json_error([
                'code' => 'not_authorized',
            ]);
            exit;
        }

        if (!isset($_POST['api_key'])) {
            wp_send_json_error([
                'code' => 'missing_parameters',
            ]);
            exit;
        }

        $apiKey = sanitize_text_field($_POST['api_key']);

        if (empty($apiKey)) {
            wp_send_json_error([
                'code' => 'missing_parameters',
            ]);
            exit;
        }

        try {
            $data = $this->getOwnerService->validateApiKeyOnApplication($apiKey);

            if (isset($data['code'])) {
                wp_send_json_error([
                    'code' => 'api_key_invalid',
                ]);
                return;
            }

			if(!isset($data['result'])){
				wp_send_json_success([
                    'user' => null,
                    'api_key' => $apiKey,
                    'project_id' => null
                ]);

                return;
			}

            $owner = $data['result'];

			if(isset($owner['total_projects']) && isset($owner['limit_projects'])){
				if($owner['total_projects'] >= $owner['limit_projects']){
					wp_send_json_success([
						'user' => $owner,
						'api_key' => $apiKey,
						'projects' => null,
						'code' => 'limit_excedeed',
					]);
					return;
				}
			}

            if ($data && !isset($data['code']) && isset($data['result']['project']['id'])) {
                wp_send_json_success([
                    'code' => 'success',
                    'project_id' => $data['result']['project']['id']
                ]);

                return;
            } elseif (!isset($data['result']['project']['id'])) {
                wp_send_json_success([
                    'code' => 'project_not_exist',
                    'project_id' => null
                ]);
            }
        } catch (\Exception $e) {
            \wp_umbrella_get_service('Logger')->error($e->getMessage());
            wp_send_json_error([
                'code' => 'unknown_error',
            ]);
            exit;
        }
    }
}
