<?php
namespace WPUmbrella\Core\Backup\Source\V2;

use WPUmbrella\Models\Backup\BackupSource;
use WPUmbrella\Models\Backup\BackupProcessCommandLine;
use Coderatio\SimpleBackup\SimpleBackup;

class MySqlManualByTableSource implements BackupSource, BackupProcessCommandLine
{
    const DEFAULT_USER = 'root';
    const DEFAULT_TIMEOUT = 900;

    protected $name;
    protected $database;
    protected $host;
    protected $user;
    protected $password;
    protected $timeout;

    /**
     * @param string      $name
     * @param string      $database
     * @param array       $databaseConnexion
     * @param int         $timeout
     */
    public function __construct($namer, $database, $databaseConnexion = [], $timeout = self::DEFAULT_TIMEOUT)
    {
		$this->version = 'v1';
        $this->namer = $namer;
        $this->database = $database;
        $this->host = isset($databaseConnexion['host']) ? $databaseConnexion['host'] : self::DEFAULT_USER;
        $this->user = isset($databaseConnexion['user']) ? $databaseConnexion['user'] : null;
        $this->password = isset($databaseConnexion['password']) ? $databaseConnexion['password'] : null;
        $this->timeout = $timeout;
    }

    public function getTimeout()
    {
        return $this->timeout;
    }

    public function getCommandLine()
    {
        return null;
    }

	public function setVersion($version){
		$this->version = $version;
		return $this;
	}

    /**
     * {@inheritdoc}
     */
    public function fetch($scratchDir)
    {

		if($this->version === 'v1'){
			$dataObj = wp_umbrella_get_service('BackupManageProcess')->getBackupData();
		}
		else if($this->version === 'v3'){
			$dataObj = wp_umbrella_get_service('BackupManageProcessCustomTable')->getBackupData();
		}

        try {
            $destination = sprintf('%s/%s/tables', $scratchDir, $this->getName());

            if (!\file_exists($destination)) {
                @\mkdir($destination, 0777, true);
            }

            $currentTable = $dataObj->getTableByCurrentBatch();
			$currentIterator = $dataObj->getBatchIterator('database');

            if ($currentTable === null) {
                return [
					'success' => true,
                    'iterator_position' => ++$currentIterator,
                    'part' => 0
                ];
            }


			$simpleBackup = SimpleBackup::setDatabase([
				$this->database,
				$this->user,
				$this->password,
				$this->host
			]);


			$tableName = $currentTable['name'];
            $simpleBackup->includeOnly([$tableName]);

			$newPart = 0;

			$filenameExport = sprintf('%s', $tableName);

			$batchs = [];
			if($dataObj->hasTableNeedBatchByName($tableName)){
				$batchs = $dataObj->getTableBatchsByName($tableName);
				$part = $dataObj->getBatchPart('database');

				if(isset($batchs[$part])){
					$newPart = $part+1;
					$offset = $batchs[$part]['offset'];
					$limit = $batchs[$part]['limit'];

					$needDropTable = $part === 0 ? true : false;

					$simpleBackup->setTableLimitsOn([
						$tableName => sprintf('%s, %s', $offset, $limit)
					]);

					$filenameExport .= sprintf('-part-%s', $part);
				}
				else{
					return [
						'success' => true,
						'iterator_position' => $currentIterator,
						'part' => $newPart
					];
				}
			}

			$simpleBackup->setAddDropTable(
				$needDropTable
			);

			$simpleBackup->storeAfterExportTo($destination, $filenameExport);

			if(!empty($batchs) && !isset($batchs[$newPart])){
				$newPart = 0;
			}

			if($newPart === 0){
				$currentIterator++;
			}


			return [
				'success' => true,
				'iterator_position' => $currentIterator,
				'part' => $newPart
			];

        } catch (\Exception $e) {
            $currentIterator = $dataObj->getBatchIterator('database');
			return [
				'success' => false,
				'iterator_position' => ++$currentIterator,
				'part' => 0
			];
        }

    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->namer->getName();
    }
}
