<?php
namespace WPUmbrella\Core;

use WPUmbrella\Core\Container\ContainerSkypress;
use WPUmbrella\Core\Container\ManageContainer;
use WPUmbrella\Core\Hooks\ActivationHook;
use WPUmbrella\Core\Hooks\DeactivationHook;
use WPUmbrella\Core\Hooks\ExecuteHooks;
use WPUmbrella\Core\Hooks\ExecuteHooksBackend;
use WPUmbrella\Core\Hooks\ExecuteHooksFrontend;
use WPUmbrella\Core\Controllers;
use WPUmbrella\Core\UmbrellaRequest;

abstract class Kernel
{
    protected static $container = null;

	protected static $universalProcess = null;


    protected static $data = ['slug' => null, 'main_file' => null, 'file' => null, 'root' => ''];

    public static function setSettings($data)
    {
        self::$data = array_merge(self::$data, $data);
    }

    public static function setContainer(ManageContainer $container)
    {
        self::$container = self::getDefaultContainer();
    }

    protected static function getDefaultContainer()
    {
        return new ContainerSkypress();
    }

    public static function getContainer()
    {
        if (null === self::$container) {
            self::$container = self::getDefaultContainer();
        }

        return self::$container;
    }

    public static function handleHooksPlugin()
    {
        require_once WP_UMBRELLA_DIR . '/src/Async/ActionSchedulerSendErrors.php';
        require_once WP_UMBRELLA_DIR . '/src/Async/ActionSchedulerSnapshotData.php';

        switch (current_filter()) {
            case 'plugins_loaded':
				if(self::$universalProcess && self::$universalProcess['load']){
					self::buildActionScheduler();

					wp_umbrella_get_service('RequestSettings')->buildRequestSettings();
				}

                load_plugin_textdomain('wp-health', false, WP_UMBRELLA_LANGUAGES);

                foreach (self::getContainer()->getActions() as $key => $class) {
                    if (!class_exists($class)) {
                        continue;
                    }

                    $class = new $class();

                    switch (true) {
                        case $class instanceof ExecuteHooksBackend:
                            if (is_admin()) {
                                $class->hooks();
                            }
                            break;

                        case $class instanceof ExecuteHooksFrontend:
                            if (!is_admin()) {
                                $class->hooks();
                            }
                            break;

                        case $class instanceof ExecuteHooks:
                            $class->hooks();
                            break;
                    }
                }
                break;
            case 'activate_' . self::$data['slug'] . '/' . self::$data['slug'] . '.php':
                foreach (self::getContainer()->getActions() as $key => $class) {
                    if (!class_exists($class)) {
                        continue;
                    }

                    $class = new $class();
                    if ($class instanceof ActivationHook) {
                        $class->activate();
                    }
                }
                break;
            case 'deactivate_' . self::$data['slug'] . '/' . self::$data['slug'] . '.php':
                foreach (self::getContainer()->getActions() as $key => $class) {
                    if (!class_exists($class)) {
                        continue;
                    }

                    $class = new $class();
                    if ($class instanceof DeactivationHook) {
                        $class->deactivate();
                    }
                }
                break;
        }
    }

    /**
     * @static
     *
     * @param string $path
     * @param string $type
     * @param string $namespace
     *
     * @return void
     */
    public static function buildClasses($path, $type, $namespace = '')
    {
        try {
            $files = array_diff(scandir($path), ['..', '.']);
            foreach ($files as $filename) {
                $pathCheck = $path . '/' . $filename;

                if (is_dir($pathCheck)) {
                    self::buildClasses($pathCheck, $type, $namespace . $filename . '\\');
                    continue;
                }

                $pathinfo = pathinfo($filename);
                if (isset($pathinfo['extension']) && 'php' !== $pathinfo['extension']) {
                    continue;
                }

                $data = '\\WPUmbrella\\' . $namespace . str_replace('.php', '', $filename);

                switch ($type) {
                    case 'services':
                        self::getContainer()->setService($data);
                        break;
                    case 'actions':
                        self::getContainer()->setAction($data);
                        break;
                }
            }
        } catch (\Exception $e) {
        }
    }

    /**
     * Build Container.
     */
    public static function buildContainer()
    {
        self::buildClasses(self::$data['root'] . '/src/Services', 'services', 'Services\\');
        self::buildClasses(self::$data['root'] . '/src/Actions', 'actions', 'Actions\\');
    }


	/**
	 * At this stage, it is necessary to check the API key
	 */
    public static function canExecute(UmbrellaRequest $request)
    {

		$requestVersion = $request->getRequestVersion();

		$token = $request->getToken();
		$response = wp_umbrella_get_service('ApiWordPressPermission')->isTokenAuthorized($token);

		if (!isset($response['authorized']) || !$response['authorized']) {
			return false;
		}

		return true;
    }

	public static function getWPEngineCookieValue() {
        return hash("sha256", 'wpe_auth_salty_dog|'.WPE_APIKEY);
    }


	/**
	 * In WP Engine hosting only requests from logged in users with auth cookies are given filesystem
	 *  write access.
	 */
	public static function setCookies($user) {

		$user_id = $user->ID;

		$cookies = [];
		$secure  = is_ssl();
		$secure  = apply_filters( 'secure_auth_cookie', $secure, $user_id );

		if ( $secure ) {
			$auth_cookie_name = SECURE_AUTH_COOKIE;
			$scheme           = 'secure_auth';
		} else {
			$auth_cookie_name = AUTH_COOKIE;
			$scheme           = 'auth';
		}

		$expiration = time() + ( DAY_IN_SECONDS * 14 );

		$cookies[ $auth_cookie_name ] = wp_generate_auth_cookie( $user_id, $expiration, $scheme );
		$cookies[ LOGGED_IN_COOKIE ]  = wp_generate_auth_cookie( $user_id, $expiration, 'logged_in' );
		if (defined('WPE_APIKEY')) {
            $cookieValue = self::getWPEngineCookieValue();
            setcookie('wpe-auth', $cookieValue, 0, '/', '', force_ssl_admin(), true);
        }

	}

    public static function trySetupAdmin()
    {
		if(!defined('WP_UMBRELLA_IS_ADMIN')){
			define('WP_UMBRELLA_IS_ADMIN', true);
		}

        if (!defined('WP_ADMIN')) {
            define('WP_ADMIN', true);
        }

        if (!defined('WP_NETWORK_ADMIN')) {
            define('WP_NETWORK_ADMIN', false);
        }

        if (isset($_GET['import']) && !defined('WP_LOAD_IMPORTERS')) {
            define('WP_LOAD_IMPORTERS', true);
        }


        if (!function_exists('get_current_screen')) {
            include_once ABSPATH . '/wp-admin/includes/class-wp-screen.php';
            include_once ABSPATH . '/wp-admin/includes/screen.php';
        }

		try {
			$GLOBALS['hook_suffix'] = '';
			\WP_Screen::get('')->set_current_screen();
		} catch (\Exception $e) {
			// No black magic
		}

        if (!function_exists('wp_set_current_user')) {
			include_once ABSPATH . '/wp-includes/pluggable.php';
        }


        $user = wp_umbrella_get_service('UsersProvider')->getUserAdminCanBy();
        if (!$user) {
            return false;
        }

		// Authenticated user
		wp_cookie_constants();
		wp_set_current_user($user->ID, $user->user_login);
		self::setCookies($user);
    }

    protected static function buildActionScheduler(){

        if(function_exists('action_scheduler_initialize_3_dot_3_dot_0')){
            action_scheduler_initialize_3_dot_3_dot_0();
            \ActionScheduler_Versions::initialize_latest_version();
        }

        if ( ! class_exists( 'ActionScheduler_Store' ) ) {
            require_once WP_UMBRELLA_DIR . '/thirds/action-scheduler/deprecated/ActionScheduler_Store_Deprecated.php';
            require_once WP_UMBRELLA_DIR . '/thirds/action-scheduler/classes/abstracts/ActionScheduler_Store.php';
            require_once WP_UMBRELLA_DIR . '/thirds/action-scheduler/classes/abstracts/ActionScheduler_Abstract_Schema.php';
            require_once WP_UMBRELLA_DIR . '/thirds/action-scheduler/classes/schema/ActionScheduler_StoreSchema.php';
        }

        $tables = [
            \ActionScheduler_StoreSchema::ACTIONS_TABLE,
            \ActionScheduler_StoreSchema::CLAIMS_TABLE,
            \ActionScheduler_StoreSchema::GROUPS_TABLE,
        ];

        global $wpdb;
        foreach ( $tables as $table ) {
            $wpdb->tables[] = $table;
            $name           = $GLOBALS[ 'wpdb' ]->prefix . $table;
            $wpdb->$table   = $name;
        }
    }


	public static function setPluginPriority()
	{
		$pluginBasename = 'wp-health/wp-health.php';
		$activePlugins  = get_option('active_plugins');

		if (!is_array($activePlugins) || reset($activePlugins) === $pluginBasename) {
			return;
		}

		$key = array_search($pluginBasename, $activePlugins);

		if ($key === false || $key === null) {
			return;
		}

		unset($activePlugins[$key]);
		array_unshift($activePlugins, $pluginBasename);
		update_option('active_plugins', array_values($activePlugins));
	}


    public static function execute($data)
    {
        if (!class_exists('ActionScheduler')) {
            require_once WP_UMBRELLA_DIR . '/thirds/action-scheduler/action-scheduler.php';
        }

        self::setSettings($data);
        self::buildContainer();

		$request = UmbrellaRequest::createFromGlobals();

		self::$universalProcess = [
			'load' => false,
			'need_delay' => false,
			'hook' => 'wp',
			'priority' => 10,
			'action' => null
		];


        if ($request->canTryExecuteWPUmbrella()) {
            $controllers = Controllers::getControllers();
            $action = $request->getAction();

            if (self::canExecute($request) && array_key_exists($action, $controllers)) {
				self::setPluginPriority();
				$item = $controllers[$action];
				self::$universalProcess = [
					'load' => true,
					'need_delay' => false,
					'action' => $action,
					'hook' => isset($item['hook']) ? $item['hook'] : 'plugins_loaded',
					'priority' => isset($item['priority']) ? $item['priority'] : 10,
				];

            }
        }

		$delayForPlugins = [];
		if(defined('WP_PLUGIN_DIR')){
			$delayForPlugins = [
				WP_PLUGIN_DIR . '/wp-marketing-automations/wp-marketing-automations.php',
				WP_PLUGIN_DIR . '/post-smtp/postman-smtp.php'
			];
		}

		$needDelayHook = false;
		foreach($delayForPlugins as $plugin){
			if(file_exists($plugin)){
				self::$universalProcess['need_delay'] = true;
				break;
			}
		}

		if(self::$universalProcess['load']){
			add_filter('deprecated_function_trigger_error', '__return_false');

			if(self::$universalProcess['need_delay']){
				add_action('plugins_loaded', function(){
					self::trySetupAdmin();
				}, 1);
			} else {
				self::trySetupAdmin();
			}

			add_action('wp_loaded', function() {
				// Prevent for WooCommerce error
				wp_umbrella_get_service('RequestSettings')->adminLoaded();

			});
			add_action('wp', function(){
				define('WP_UMBRELLA_FALLBACK_REQUEST', true);
				require_once WP_UMBRELLA_DIR . '/request/umbrella-application.php';
				return;
			}, self::$universalProcess['priority']);
		}

        add_action('plugins_loaded', [__CLASS__, 'handleHooksPlugin'], 10);
        register_activation_hook($data['file'], [__CLASS__, 'handleHooksPlugin']);
        register_deactivation_hook($data['file'], [__CLASS__, 'handleHooksPlugin']);
    }
}

